// Package Includes
const ws = require('ws');
const fs = require('fs');
const http = require('http');
const https = require('https');
const crypto = require('crypto');
const express = require('express');
const { exec, execSync } = require('child_process');
const scanner = require('node-wifi-scanner');
const multer = require('multer');
const path = require('path');
const protobuf = require("protobufjs");
const os = require('os');
const readline = require('readline');
const AdmZip = require('adm-zip');
const morgan = require('morgan');

// Custom Packages
let sd = require('./server_data');
const { logLevels, console_log_warn, console_log_blue, console_log_error, console_log_ok, console_log_cyan } = require('./server_util');

protobuf.load(path.join(__dirname, 'Vega.proto')).then((vegaRoot) => {
	FontTypes = vegaRoot.lookupEnum('vega.Font.Style')["values"];
});
protobuf.load(path.join(__dirname, 'Network.proto')).then((networkRoot) => {
	StatusCodes = networkRoot.lookupEnum('vega.StatusCodes')["values"];
});

// Constant variables 
const angularPath = path.join(__dirname, 'public', 'ui');
const uploadPath = path.join(__dirname, 'uploads');

// Editable variables
let wifiNetworks = [];

// Configure Express
const app = express();
app.use('/api/modsConfigured', express.json());
app.use('/api/modsConfiguredNode', express.json());
app.use('/api/displayConfig', express.json());
app.use('/api/modStatus', express.json());

app.use(express.static(angularPath));
app.use(express.static(path.join(__dirname, '/public')));

// Increase the allowed HTTP payload size (defaults to 1000 params and 100kb) so that we can configure larger networks.
// parameterLimit and limit values were decided based on vaues found/recommended online. Can be changed in the future if necessary.
app.use(express.urlencoded({ parameterLimit: 100000, limit: '50mb', extended: true }));


// Add access logging
app.use(morgan(function (tokens, req, res) {
    let remote_addr = tokens['remote-addr'](req) || "unknown";
    let remote_user = tokens['remote-user'](req) || "unknown";
    let content_length = tokens.res(req, 'content-length') || res.get('Content-Length') || "unknown";

    let logString = `remote-addr: ${remote_addr} - remote-user: ${remote_user} - `+
    `${tokens.method(req, res)} ${tokens.url(req, res)} ${tokens.status(req, res)} - `+
    `content_length: ${content_length}`;
    logToBackend(logString, logLevels.Extreme, false);
    //return logString;
}));

// Check if the uploads directory exists if it does not create it
if (!fs.existsSync(uploadPath)) {
    fs.mkdirSync(uploadPath);
}
// Configure multer for file storage
const storage = multer.diskStorage({
    destination: (req, file, cb) => cb(null, uploadPath),
    filename: (req, file, cb) => cb(null, file.originalname)
});
const upload = multer({ storage });

const options = {
	key: fs.readFileSync('/usr/share/vega/ui/https_certs/client-key.pem'),
	cert: fs.readFileSync('/usr/share/vega/ui/https_certs/client-cert.pem'),
};

const server = https.createServer(options, app);

// Configure Server
const serverPortNumber = 443;
server.listen(serverPortNumber, () => {
	console.log('HTTPS Server running on Port ' + serverPortNumber);
});
server.setTimeout(200000);

// configure http for port forwarding
const http_app = express();
http_app.use(function(req, res) {
    let domain = req.get('host');
    res.redirect('https://' + domain + req.originalUrl);
});
const http_server = http.createServer(http_app);
const httpServerPortNumber = 80;
http_server.listen(httpServerPortNumber, () => {
	console.log('HTTP Server running on Port ' + httpServerPortNumber);
});
http_server.setTimeout(200000);

// ------- Init Parameters  ------- 
const wss = new ws.Server({ server });

sd.connectToBackend(__dirname, wss);
sd.getVersionInfo();

sd.getVersionInfo();

async function logToBackend(msg, level, log_to_ui = true) {
    sd.logToUI(msg, true, level, log_to_ui);

    await sd.putDataZmq(
        '/log_to_file',
        { 
            "value": level,
            "description": "NodeJS - " + msg 
        }
    );
}

// Middleware to verify that the FL5 Backend is running
const verifyFL5Backend = async (req, res, next) => {
	const timeoutMs = 5000;
	const intervalMs = 100;
	const iterations = Math.floor(timeoutMs / intervalMs);
	let pingData = sd.getPingData();

	pingData.response = false;
	sd.setPingData(pingData);
	await sd.getDataZmq('/ping');

	for (let idx = 0; idx < iterations && !pingData.response; ++idx) {
		await new Promise(resolve => setTimeout(resolve, intervalMs));
		pingData = sd.getPingData();

		if (pingData.response) {
			break;
		}
	}

	if (!pingData.response) {
		logToBackend(`verifyFL5Backend() - Could not contact FL5 backend for ${req.path}. Is it running?`, logLevels.Extreme);

		return res.status(500).send('Error occurred in middleware');
	} else {
		next();
	}
};
//app.use(verifyFL5Backend);

   

// ------- Page Data Request/Responses ------- 
app.get('/', (req, res) => {
	res.sendFile(path.join(angularPath, 'index.html'));
});

app.post('/api/getLogFiles', (req, res) => {
    let requestData = '';

    req.on('data', (chunk) => {
        requestData += chunk.toString();
    });

	req.on('end', async () => {
        const date = new Date();
        const zip = new AdmZip();

        let log_files = fs.readdirSync(sd.logDir, { withFileTypes: true })
            .filter(dirent => dirent.isFile())
            .map(dirent => dirent.name);
        log_files = log_files.filter((file) => { return file.endsWith(".log") });
        for (let idx = 0; idx < log_files.length; ++idx) {
            zip.addFile(`logs/${log_files[idx]}`, fs.readFileSync(`${sd.logDir}/${log_files[idx]}`), '', 0o0644);
        }

        zip.addFile("logs/log_terminal.txt", Buffer.from(requestData, "utf8"), '', 0o0644);

        const config_files = fs.readdirSync(sd.configDir, { withFileTypes: true })
            .filter(dirent => dirent.isFile())
            .map(dirent => dirent.name);
        for (let idx = 0; idx < config_files.length; ++idx) {
            zip.addFile(`config/${config_files[idx]}`, fs.readFileSync(`${sd.configDir}/${config_files[idx]}`), '', 0o0644);
        }

        const proto_files = fs.readdirSync(sd.protoDir, { withFileTypes: true })
            .filter(dirent => dirent.isFile())
            .map(dirent => dirent.name);
        for (let idx = 0; idx < proto_files.length; ++idx) {
            zip.addFile(`proto/${proto_files[idx]}`, fs.readFileSync(`${sd.protoDir}/${proto_files[idx]}`), '', 0o0644);
        }

        sd.logToUI("Retrieved Log Files", true, logLevels.Debug, true);

        const dateStr = `${date.getMonth() + 1}/${date.getDate()}/${date.getFullYear()}-${date.getHours()}:${date.getMinutes()}:${date.getSeconds()}`;
        const downloadName = `fl5_logs_${dateStr}.zip`;
        const res_data = zip.toBuffer();
        res.set('Content-Type','application/octet-stream');
        res.set('Content-Disposition',`attachment; filename=${downloadName}`);
        res.set('Content-Length', res_data.length);
        res.send(res_data);
	});
});

app.get('/api/getProductsFL5', async (req, res) => {
    let fl5_products;
    let price_format;
	await sd.getDataZmq('/fl5_products').then( data => { fl5_products = data.products; });
	await sd.getDataZmq('/format').then( data => { price_format = data; });
    if (!fl5_products || !price_format) {
        res.send();
        return;
    }

	let data = new Object();
	let productArray = [];
	for (let i = 0; i < fl5_products.length; i++) {
		let productObject = new Object();
		productObject.id = fl5_products[i].id;
		productObject.name = fl5_products[i].name;
		productObject.offset = fl5_products[i].offset;
		productObject.neg_offset = fl5_products[i].negOffset;
		productObject.pct_offset = fl5_products[i].pctOffset;
		productObject.neg_pct_offset = fl5_products[i].negPctOffset;
		productObject.price = fl5_products[i].displayPrice;
		productObject.pay_type = fl5_products[i].payType;
		productObject.service_type = fl5_products[i].serviceType;
		productArray.push(productObject);
	}
	data.products = productArray;

	let jsonString = JSON.stringify(data);
	res.send(jsonString);
});

app.get('/api/getNetwork', async (req, res) => {
    let wifiSSID;
    let wifiPassword;

	await sd.getDataZmq('/wifi').then( data => { 
        wifiSSID = data.ssid;
        wifiPassword = data.pass;
    });

	let data = new Object();
	let networkObject = new Object();
	networkObject.connectionStatus = getWiFiStatus(wifiSSID);
	networkObject.wifiSSID = wifiSSID;
	networkObject.wifiPassword = wifiPassword;
	networkObject.ipAdresses = getIpAddresses();
	data.network = networkObject;

	let jsonString = JSON.stringify(data);
	res.send(jsonString);
});

app.put('/api/wifiChange', (req, res) => {
	req.on('data', async (data) => {
		let wifiData = JSON.parse(data.toString());
		logToBackend(`Wifi settings update: ${JSON.stringify({ ssid: wifiData.ssid, pass: (wifiData.password.length === 0)?"":"[hidden]" }, null, "")}`, logLevels.Info);
		const success = update_wpa_configuration(wifiData.ssid, wifiData.password);
        if (success) {
            await sd.putDataZmq('/wifi', { ssid: wifiData.ssid, pass: "" });
            console.log("Wifi change initiated, sending response");
        }
        res.send(JSON.stringify({
            success: success
        }));
	});
});

app.get('/api/networkStatus', async (req, res) => {
	await sd.getDataZmq('/network_status');
    res.send();
});

app.get('/api/cachedWifiNetworks', async (req, res) => {
    let wifiNetworks = sd.getWifiNetworks();
    console.log("wifiNetworks", wifiNetworks);
    let jsonString = JSON.stringify(wifiNetworks);
    console.log("jsonString", jsonString);
    res.send(jsonString);
});

app.get('/api/getNetworkScan', (req, res) => {
	return scanner.scan((err, networks) => {
		if (err) {
			console.error(err);
			return;
		}
		else {
			wifiNetworks = [];
			for (connection of networks) {
				if (connection.ssid != "" && wifiNetworks.indexOf(connection.ssid) == -1) {
					wifiNetworks.push(connection.ssid);
				}
			}
            sd.setWifiNetworks(wifiNetworks);
            console.log("Found networks:");
            console.log(wifiNetworks);
            let jsonString = JSON.stringify(wifiNetworks);
            res.send(jsonString);
		}
	});
});

app.put('/api/priceChange', (req, res) => {
	req.on('data', async function (data) {
		let priceDataUI = JSON.parse(data.toString());
		let priceDataBackend;
        let products;
        await sd.getDataZmq('/fl3_products').then( (fl3_prod_data) => {  products = fl3_prod_data; });
        await sd.getDataZmq('/prices').then( (price_data) => {  priceDataBackend = price_data; });
		logToBackend(`Price update: ${JSON.stringify(priceDataUI, null, "")}`, logLevels.Info);

		for (let priceLine = 0; priceLine < priceDataUI.prices.length; ++priceLine) {
			let newPrice = priceDataUI.prices[priceLine];
			if (newPrice != "disabled") {
				let prodID = products[priceLine].id;
				for (let idx = 0; idx < priceDataBackend.length; ++idx) {
					if (parseInt(priceDataBackend[idx].productId) == parseInt(prodID)) {
						priceDataBackend[idx].setPrice = newPrice;
					}
                }
			}
		}
		const backendPriceData = {
			prices: priceDataBackend
		}

        await sd.putDataZmq('/prices', backendPriceData);
	});
	res.send();
});

app.put('/api/fl3ProductChange', (req, res) => {
	req.on('data', async function (data) {
		let productDataUI = JSON.parse(data.toString());
		let priceDataBackend;
        let productDataBackend;
        await sd.getDataZmq('/fl3_products').then( (fl3_prod_data) => {  priceDataBackend = fl3_prod_data; });
        await sd.getDataZmq('/prices').then( (price_data) => {  priceDataBackend = price_data; });

		logToBackend(`fl3ProductChange: ${JSON.stringify(productDataUI, null, "")}`, logLevels.Info);
		for (let i = 0; i < productDataUI.products.length; i++) {
			if (productDataBackend.id.toLowerCase() != "none") {
				productDataBackend[i].id = productDataUI.products[i].id;
				priceDataBackend[i].productId = productDataUI.products[i].id;
				productDataBackend[i].name = productDataUI.products[i].name;
				productDataBackend[i].offset = productDataUI.products[i].offset;
				productDataBackend[i].negOffset = productDataUI.products[i].neg_offset;
			}
			else {
				productDataBackend[i].name = "";
				productDataBackend[i].offset = "0.00";
				productDataBackend[i].negOffset = false;
			}
		}
		const backendProductData = { products: productDataBackend };
        await sd.putDataZmq('/fl3_products', backendProductData);

		const backendPriceData = { prices: priceDataBackend }
        await sd.putDataZmq('/prices', backendPriceData);
	});
	res.send();
});

app.put('/api/fl5ProductChange', (req, res) => {
	req.on('data', async function (data) {
		let productData = JSON.parse(data.toString());

        let fl5ProductsBackend = [];
        let posMode;
        await sd.getDataZmq('/fl5_products').then( (fl5_prod_data) => {  
            if (fl5_prod_data.products) {
                fl5ProductsBackend = fl5_prod_data.products; 
            }
        });
        await sd.getDataZmq('/pos').then( (pos_data) => {  posMode = pos_data.priceMode; });

		let update_type = "product"
		if (productData.type) {
			update_type = productData.type;
		}

		let logStr = "";
		if (update_type == "price") {
			logStr = "Price Update:";
			for (let idx = 0; idx < productData.products.length; ++idx) {
				logStr += ` Prod${idx + 1}-$${productData.products[idx].price}`;
			}
		} else if (update_type == "product") {
			logStr = "Product Update:\n";
		}

        maxLength = Math.min(productData.products.length, fl5ProductsBackend.length);
		for (let idx = 0; idx < maxLength; idx++) {
			if (update_type == "product" && idx > 0) logStr += `\n`;

			if (productData.products[idx].hasOwnProperty("name")) {
				fl5ProductsBackend[idx].name = productData.products[idx].name;
				if (update_type == "product") logStr += `${productData.products[idx].name}: `
			}
			if (productData.products[idx].hasOwnProperty("id")) {
				const prod_id = productData.products[idx].id;
				if (/^\d+$/.test(prod_id) && prod_id.length <= 4 || prod_id == "none") { // Check that the ID is a string that contains a number
					fl5ProductsBackend[idx].id = prod_id;
				}
				if (update_type == "product") logStr += `ID-${prod_id} `;
			}
			if (productData.products[idx].hasOwnProperty("offset")) {
				fl5ProductsBackend[idx].offset = productData.products[idx].offset;
				if (update_type == "product") logStr += `offset:${productData.products[idx].offset} `
			}
			if (productData.products[idx].hasOwnProperty("neg_offset")) {
				fl5ProductsBackend[idx].negOffset = productData.products[idx].neg_offset;
				if (update_type == "product") logStr += `neg_offset:${productData.products[idx].neg_offset} `
			}
			if (productData.products[idx].hasOwnProperty("pct_offset")) {
				fl5ProductsBackend[idx].pctOffset = productData.products[idx].pct_offset;
				if (update_type == "product") logStr += `pct_offset:${productData.products[idx].pct_offset} `
			}
			if (productData.products[idx].hasOwnProperty("neg_pct_offset")) {
				fl5ProductsBackend[idx].negPctOffset = productData.products[idx].neg_pct_offset;
				if (update_type == "product") logStr += `neg_pct_offset:${productData.products[idx].neg_pct_offset} `
			}
			if (productData.products[idx].hasOwnProperty("price")) {
                // We only get here if we modify prices in the Prices page
				fl5ProductsBackend[idx].displayPrice = productData.products[idx].price;
                // Undo the offset for the "raw" price
                const offModifier = fl5ProductsBackend[idx].negOffset ? -1 : 1;
                const pctOffModifier = fl5ProductsBackend[idx].negPctOffset ? -1 : 1;
                const offset = Number(fl5ProductsBackend[idx].offset) * offModifier;
                const pctOffset = (Number(fl5ProductsBackend[idx].pctOffset) * pctOffModifier) / 100;
                const price = Number(productData.products[idx].price);
                let rawPrice;

                // Calculate what the price would have to be in order to end up with the user submitted price after the offsets are factored in
                if (1 + pctOffset == 0) {
                    rawPrice = 0;
                } else if (1 + pctOffset == 2) {
                    rawPrice = (price - offset) * (1 + pctOffset);
                } else {
                    rawPrice = (price - offset) / (1 + pctOffset);
                }

                rawPrice = rawPrice.toFixed(3); // Set a fixed number of decimal places (fixes floating point precision issues)

                fl5ProductsBackend[idx].price = rawPrice.toString();
				if (update_type == "product") logStr += `price:${productData.products[idx].price} `
			}
            else {
                // This is the case for modifying the product config
                if (posMode > 0) {
                    const offModifier = fl5ProductsBackend[idx].negOffset ? -1 : 1;
                    const pctOffModifier = fl5ProductsBackend[idx].negPctOffset ? -1 : 1;
                    const offset = Number(fl5ProductsBackend[idx].offset) * offModifier;
                    const pctOffset = (Number(fl5ProductsBackend[idx].pctOffset) * pctOffModifier) / 100;
                    const price = Number(fl5ProductsBackend[idx].price);
                    let rawPrice;

                    // Calculate the price after the offset is factored in
                    if (1 + pctOffset == 0) {
                        rawPrice = 0;
                    } else if (1 + pctOffset == 2) {
                        rawPrice = (price / 2) + offset;
                    } else {
                        rawPrice = (price + (price * pctOffset)) + offset;
                    }

                    rawPrice = rawPrice.toFixed(3); // Set a fixed number of decimal places (fixes floating point precision issues)

                    fl5ProductsBackend[idx].displayPrice = rawPrice.toString();
                }
            }
			if (productData.products[idx].hasOwnProperty("pay_type")) {
				fl5ProductsBackend[idx].payType = productData.products[idx].pay_type;
				if (update_type == "product") logStr += `pay_type:${productData.products[idx].pay_type} `
			}
			if (productData.products[idx].hasOwnProperty("service_type")) {
				fl5ProductsBackend[idx].serviceType = productData.products[idx].service_type;
				if (update_type == "product") logStr += `service_type:${productData.products[idx].service_type} `
			}
		}
        console_log_warn("-----------------------------------------------------");

		logToBackend(`${logStr}`, logLevels.Info);

        await sd.putDataZmq('/fl5_products', { products: fl5ProductsBackend });
	});
	res.send();
});

app.get('/api/POS_backend', async (req, res) => {
    let priceMode;
    await sd.getDataZmq('/pos').then( (pos_status) => {  priceMode = pos_status.priceMode; });

	res.send(JSON.stringify({
		price_mode: priceMode
	}));
});

app.put('/api/POS_backend', (req, res) => {
	req.on('data', async function (data) {
		let pos_data = JSON.parse(data.toString());
		logToBackend(`POS Update - Name: ${pos_data.name} - Type ID: ${pos_data.price_mode}`, logLevels.Info);
        await sd.putDataZmq('/pos', { price_mode: pos_data.price_mode });
	});

	res.send(JSON.stringify({
		success: true
	}));
});

// Support function for PUT /api/modsConfigured
function setNonRecursiveNodeProps(updatedNode, incomingNode, cabAssyConfigs, frameConfig) {
	let numFrames = 1;
	if (frameConfig.frames) {
		numFrames = frameConfig.frames.length
	}

	let emptyFS = [];
	for (let fs_idx = 0; fs_idx < numFrames; ++fs_idx) {
		emptyFS.push({ "message": -1, "frame": fs_idx + 1 });
	}

	updatedNode.status = 0;
	updatedNode.uuid = incomingNode.hasOwnProperty("uuid") ? incomingNode.uuid : "";

	if (incomingNode.hasOwnProperty("status") && incomingNode.status) {
        updatedNode.status = StatusCodes[0];
		if (StatusCodes.hasOwnProperty(incomingNode.status)) {
            updatedNode.status = StatusCodes[incomingNode.status];
		} else if (Object.values(StatusCodes).includes(incomingNode.status)) {
			updatedNode.status = incomingNode.status;
		}
	}
	if (incomingNode.type == "server") {
		updatedNode.networkId = incomingNode.hasOwnProperty("networkId") ? incomingNode.networkId : "";
		updatedNode.channel = incomingNode.hasOwnProperty("channel") ? incomingNode.channel : "";
    }
	if (incomingNode.type == "server" || incomingNode.type == "direct") {
		updatedNode.port = incomingNode.hasOwnProperty("port") ? incomingNode.port : 0;
		updatedNode.fl3 = incomingNode.hasOwnProperty("fl3") ? incomingNode.fl3 : false;
	}
	if (incomingNode.type == "server" || incomingNode.type == "client") {
		updatedNode.firmwareVersion = incomingNode.hasOwnProperty("firmwareVersion") ? incomingNode.firmwareVersion : "";
		updatedNode.hardwareVersion = incomingNode.hasOwnProperty("hardwareVersion") ? incomingNode.hardwareVersion : "";
	}
	if (incomingNode.type != "module") {
		updatedNode.name = incomingNode.hasOwnProperty("name") ? incomingNode.name : "";
		updatedNode.error = incomingNode.hasOwnProperty("error") ? incomingNode.error : false;
	}
	if (incomingNode.type == "display") {
		updatedNode.subtype = incomingNode.hasOwnProperty("subtype") ? incomingNode.subtype : "price";

		updatedNode.frameSelect = incomingNode.hasOwnProperty("frameSelect") && incomingNode.frameSelect.length > 0 ? [...incomingNode.frameSelect] : emptyFS;

		if (incomingNode.hasOwnProperty("children") && incomingNode.children.length > 0) {
			const numPanels = incomingNode.children;
			const hwId = incomingNode.children[0];
			const cabAssyConf = cabAssyConfigs.filter((conf) => {
				return hwId == conf["hwId"] && numPanels == conf["panels"];
			});

			updatedNode.assyConfig = incomingNode.hasOwnProperty("assyConfig") ? incomingNode.assyConfig : cabAssyConf[0];
		}
	}
	if (incomingNode.type == "module") {
		updatedNode.hwId = incomingNode.hasOwnProperty("hwId") ? incomingNode.hwId : "";
		updatedNode.linearAddr = incomingNode.hasOwnProperty("linearAddr") ? incomingNode.linearAddr : updatedNode.linearAddr;
		updatedNode.nodeAddr = incomingNode.hasOwnProperty("nodeAddr") ? incomingNode.nodeAddr : updatedNode.nodeAddr;
		updatedNode.assyAddr = incomingNode.hasOwnProperty("assyAddr") ? incomingNode.assyAddr : updatedNode.assyAddr;
		updatedNode.assyId = incomingNode.hasOwnProperty("assyId") ? incomingNode.assyId : updatedNode.assyId;
		updatedNode.fwRev = incomingNode.hasOwnProperty("fwRev") ? incomingNode.fwRev : updatedNode.fwRev;
		updatedNode.fontRev = incomingNode.hasOwnProperty("fontRev") ? incomingNode.fontRev : updatedNode.fontRev;
		updatedNode.backup_comms = incomingNode.hasOwnProperty("backup_comms") ? incomingNode.backup_comms : false;
		if (incomingNode.hasOwnProperty("extStatus") && updatedNode.hasOwnProperty("extStatus")) {
			if (incomingNode.extStatus.hasOwnProperty("dimmingConfig") && updatedNode.extStatus.hasOwnProperty("dimmingConfig")) {
				updatedNode.extStatus.dimmingConfig.dimmingCalibration = incomingNode.extStatus.dimmingConfig.hasOwnProperty("dimmingCalibration") ? incomingNode.extStatus.dimmingConfig.dimmingCalibration : updatedNode.extStatus.dimmingConfig.dimmingCalibration;
			}
		}
	}
}

// Support function for PUT /api/modsConfigured
function dispNodesUiToProtobuf(uiNodes, cabAssyConfigs, frameConfig) {

	let protoNodes = [];

	for (let idx = 0; idx < uiNodes.length; ++idx) {
		const uiNode = uiNodes[idx];
		let protoNodeChldrn = [];

		if (uiNode.children && uiNode.children.length > 0) {
			protoNodeChldrn = dispNodesUiToProtobuf(uiNode.children, cabAssyConfigs, frameConfig);
		}

		// Try to find the mod by UUID
		//   -- If found then assign it
		//   -- If not found then I need to create a new module and add it to mods configured
		let protoNode = {};
		if (uiNode.type === "module" && uiNode.uuid === "") {
			const modUUID = crypto.randomUUID();
			uiNode.uuid = modUUID;
			protoNode = {
				"module": {
					"uuid": modUUID,
					"hwId": uiNode.hwId,
					"status": uiNode.status ? uiNode.status : "ok"
				}
			}
			if (uiNode.hasOwnProperty("linearAddr") && uiNode.linearAddr) {
				protoNode.module.linearAddr = uiNode.linearAddr;
			}
		} else {
			protoNode = { [uiNode.type]: {} };
		}
		if (JSON.stringify(protoNode) == "{}") {
			protoNode = { [uiNode.type]: {} }
		}
		setNonRecursiveNodeProps(protoNode[uiNode.type], uiNode, cabAssyConfigs, frameConfig);

		if (protoNodeChldrn.length > 0) {
			protoNode[uiNode.type].children = protoNodeChldrn;
		}
		protoNodes.push(protoNode);
	}

	return protoNodes;
}

// Support function for PUT /api/modsConfigured
function getProtoNode(incomingNode, cabAssyConfigs, frameConfig) {
	let updatedNode = {}

	setNonRecursiveNodeProps(updatedNode, incomingNode, cabAssyConfigs, frameConfig);
	if (incomingNode.hasOwnProperty("children")) {
		let children = dispNodesUiToProtobuf(incomingNode.children, cabAssyConfigs, frameConfig);
		updatedNode.children = [...children];
	}
	protoNode = { [incomingNode.type]: updatedNode };
	return protoNode
}

/////////////////////////////////////////////////
// endpoint: /api/modsConfigured
// method: PUT
// description: Update a node in the displays portion of the config
// params:
//    data - a dictionary with the following key / value pairs
//       name: string
//       type: string
//       error: boolean
//       status: integer (corresponding to values in statusCodes)
//       firmwareVersion: string (if type server or client)
//       hardwareVersion: string (if type server or client)
//       subtype: integer (if type display, corresponding to values in displaySubtypes)
/////////////////////////////////////////////////
app.put('/api/modsConfigured', async (req, res) => {
    let cabinetAssemblyConfigurations;
    let frames;

	await sd.getDataZmq('/cabinet_assembly_configurations').then( data => { cabinetAssemblyConfigurations = data.cabinetAssemblyConfigurations; });
	await sd.getDataZmq('/frames').then( data => { frames = data; });

	const userConfiguration = req.body.config;
	const pendingConfiguration = req.body.pending;
	let modsConfigured = [];

	if (!userConfiguration) { return; }
	
	if (Array.isArray(userConfiguration)) {
		for (let idx = 0; idx < userConfiguration.length; ++idx) {
			modsConfigured.push(getProtoNode(userConfiguration[idx], cabinetAssemblyConfigurations, frames));
		}
	} else {
		modsConfigured.push(getProtoNode(userConfiguration, cabinetAssemblyConfigurations, frames));
	}
	// Write the changes to the global config
    if (pendingConfiguration) {
        logToBackend(`Submitting mod configuration to /pending_mod_configuration`, logLevels.Verbose);

        await sd.putDataZmq('/pending_mod_configuration', { "pendingModConfiguration": modsConfigured });
        sd.logToUI("Updating pending display configuration", true, logLevels.Error, false);
    } else {
        logToBackend(`Submitting mod configuration to /mods_configured`, logLevels.Verbose);
        await sd.putDataZmq('/mods_configured', { "modsConfigured": modsConfigured });
        sd.logToUI("Updating display configuration", true, logLevels.Error, false);
    }

	res.send({
		"success": true
	});
});

app.get('/api/modsConfigured', async (req, res) => {
	if (!res) {
		res.send([]);
		return;
	}

	sd.getDataZmq('/mods_configured').then( (modsConfigured) => {
        // The length condition may actually conflict with the defaults in the fl5 controller. Keep an eye on it
        if (!modsConfigured || JSON.stringify(modsConfigured) == '{}' || modsConfigured.modsConfigured.length == 0) {
            if (!modsConfigured) {
                logToBackend(`GET /api/modsConfigured: Could not get configured displays from backend , mods configured not defined`, logLevels.Error, false);
                sd.logToUI("Unable to retrieve configured displays.", true, logLevels.Error);
            } else if (JSON.stringify(modsConfigured) == '{}') {
                logToBackend(`GET /api/modsConfigured: Could not get configured displays from backend, mods configured empty`, logLevels.Error, false);
                sd.logToUI("No configured display found.", true, logLevels.Error);
            } else if (modsConfigured.modsConfigured.length == 0) {
                logToBackend(`GET /api/modsConfigured: Could not get configured displays from backend, no devices in mods configured`, logLevels.Error, false);
                sd.logToUI("No configured display found.", true, logLevels.Error);
            }
            res.send([]);
            return;
        }

        res.send(modsConfigured["modsConfigured"]);
    });
});

app.put('/api/resubmitMods', async (req, res) => {
    let modsConfigured;
    // Pulling mods configured and resubmitting isn't necessicary as of 11/25/24 but it may be in the future 
    // if we use the onchange function for the resource, so this step is here for posterity's sake.
	await sd.getDataZmq('/mods_configured').then( data => { modsConfigured = data; });
    await sd.putDataZmq('/mods_configured', modsConfigured);

    await sd.putDataZmq('/configure_network', { "command": "configure" });
	res.send({ "success": true });
})

// Support function for PUT /api/modsConfiguredNode
function findDispNodeInModsConfigured(modsConfigured, uuid) {
	let nodeFound = null;

	if (!modsConfigured) { return null; }

	for (let idx = 0; idx < modsConfigured.length; ++idx) {
		let key = Object.keys(modsConfigured[idx])[0];
		const data = modsConfigured[idx][key];

		if (!data.hasOwnProperty("uuid")) { return null; }
		if (data.uuid === uuid) {
			nodeFound = data;
		} else if (data.hasOwnProperty("children") && data.children.length > 0) {
			nodeFound = findDispNodeInModsConfigured(data.children, uuid);
		}
		if (nodeFound) {
			break;
		}
	}

	return nodeFound;
}

// Send the request to update info about the node in the mods configured file, will not update actual displays on the network
app.put('/api/modsConfiguredNode', async (req, res) => {
    let modsConfigured;
    let cabinetAssemblyConfigurations;
    let frames;
	let success = false;
	let node = req.body.data;
	let log_data = req.body.log_data
	let prefix = req.body.prefix
    let name = "Unnamed";
    if (node.hasOwnProperty("name") && node.name) {
        name = node.name == "" ? "Unnamed" : node.name;
    }
    if (prefix[prefix.length - 1] == ":") {
        prefix = prefix.slice(0, -1);
    }

	await sd.getDataZmq('/mods_configured').then( data => { modsConfigured = data; });
	await sd.getDataZmq('/cabinet_assembly_configurations').then( data => { cabinetAssemblyConfigurations = data.cabinetAssemblyConfigurations; });
	await sd.getDataZmq('/frames').then( data => { frames = data; });

	if (node && node.hasOwnProperty("uuid")) {
		const dispNode = findDispNodeInModsConfigured(modsConfigured.modsConfigured, node.uuid);
		if (dispNode) {
			setNonRecursiveNodeProps(dispNode, node, cabinetAssemblyConfigurations, frames);
			// Add the changes to the queue so that we dont overload the zmq req/rep server
            
            if (node.hasOwnProperty("type") && node.type == 'server' && !node.fl3) {
                await sd.putDataZmq('/rename_radio', { "success": false, "radio_uuid": "", "new_name": name, "server_port": dispNode.port });
            } else if (node.hasOwnProperty("type") && node.type == 'client') {
                let port = "";
                let fl3_port = false;
                modsConfigured.modsConfigured.forEach((server) => {
                    if (server.server) {
                        server.server.children.forEach((client) => {
                            if (client.client.uuid === node.uuid) {
                                fl3_port = server.server.fl3;
                                port = server.server.port;
                            }
                        });
                    }
                });
                if (!fl3_port) {
                    await sd.putDataZmq('/rename_radio', { "success": false, "radio_uuid": dispNode.uuid, "new_name": name, "server_port": port });
                }
            }
            await sd.putDataZmq('/mods_configured', { "modsConfigured": modsConfigured.modsConfigured });
			success = true;
			logToBackend(`Updating configuration for ${prefix}: ${name} - ${log_data}`, logLevels.Info);
		}
	}

	if (!success) {
		logToBackend(`PUT /api/modsConfiguredNode: Unable to successfully submit mods configured for node`, logLevels.Error, false);
        sd.logToUI("Failed to submit display configuration", true, logLevels.Error);
	}
	res.send({
		"success": success
	});
});

// Send the request to update the display to the display config route
app.put('/api/displayConfig', async (req, res) => {
	const data = req.body;

	let display = data["display"];
	delete display.messageOptions;
	delete display.type;
	delete display.prefixStr;
	delete display.retrievingModStatus;
	delete display.calibrationMode;
	delete display.statusDesc;
	delete display.fl3;
	delete display.modPixelPitchMm;
	delete display.modColor;
	delete display.parentError;
	display.children.forEach((child) => {
		delete child.module.parentError;
	});
	delete display.fl3Status;

    await sd.putDataZmq('/display_config', { "display": data["display"], "client_uuid": data["client_uuid"] });

	logToBackend(`Sending display update`, logLevels.Info);
	res.send({
		"success": true
	});
});

app.put('/api/configureNetwork', async (_, res) => {
    wss.clients.forEach(function (client) {
        client.send(JSON.stringify({ "topic": "status-update", "message": "Configure" }));
    });

    await sd.putDataZmq('/configure_network', { "command": "configure" });
	logToBackend(`Applying network configuration`, logLevels.Info);
	res.send({
		"success": true
	});
});

app.get('/api/hwIdGroups', async (_, res) => {
    let hwIdGroups;
	await sd.getDataZmq('/hw_id_groups').then( data => { hwIdGroups = data; });

	if (!hwIdGroups || hwIdGroups == {}) {
		console.log("Could not get hardware ID groups")
		res.send([]);
		return;
	}

	res.send(hwIdGroups);
});

app.get('/api/detectDisplays', async (req, res) => {
    wss.clients.forEach(function (client) {
        client.send(JSON.stringify({ "topic": "status-update", "message": "Detect" }));
    });

	logToBackend(`Starting detection of displays on the network`, logLevels.Info);
	await sd.getDataZmq('/detect_displays');
	res.send([]);
});

app.get('/api/modsAvailable', async (req, res) => {
	if (!res) {
		res.send(JSON.stringify({
			error: true
		}));
		return;
	}

    let modsAvailable;
	await sd.getDataZmq('/mods_available').then( data => { modsAvailable = data; });
    
	if (!modsAvailable || modsAvailable == {}) {
        logToBackend(`Could not find mods available, module information missing`, logLevels.Error, false);
        sd.logToUI("Could not find available displays, a re-detection may be required", true, logLevels.Error);
		res.send([]);
		return;
	}
    
	res.send(modsAvailable);
});

app.get('/api/appState', async (req, res) => {
    let appState = {};

	await sd.getDataZmq('/app_state').then( data => { appState = data; });

    if (appState.hasOwnProperty("state")) {
        res.send(JSON.stringify(appState.state));
    } else {
        res.send(JSON.stringify(["Starting"]));
    }
});


// ################### Bookmark ###################
app.get('/api/dimming', async (req, res) => {
    let dimmingData;
	await sd.getDataZmq('/dimming').then( data => { dimmingData = data; });

	res.send(JSON.stringify({
		dim_mode: dimmingData.mode,
		man_level: dimmingData.level,
		auto_min: dimmingData.min,
		auto_max: dimmingData.max
	}));
});

app.put('/api/dimming', async (req, res) => {
    let dimmingData;
	await sd.getDataZmq('/dimming').then( data => { dimmingData = data; });

	req.on('data', async function (data) {
		let updatedDimConfig = JSON.parse(data.toString());

		let logMsg = "\n";
		if (dimmingData.mode == 'M') {
			logMsg += `Original | Manual - ${dimmingData.level}`;
		} else {
			logMsg += `Original | Automatic - min:${dimmingData.min} max:${dimmingData.max}`;
		}
		logMsg += "\n";
		if (updatedDimConfig.dim_mode == 'M') {
			logMsg += `Updated | Manual - ${updatedDimConfig.man_level}`;
		} else {
			logMsg += `Updated | Automatic - min:${updatedDimConfig.auto_min} max:${updatedDimConfig.auto_max}`;
		}

		const backendDimmingData = {
			mode: updatedDimConfig.dim_mode,
			level: updatedDimConfig.man_level,
			min: updatedDimConfig.auto_min,
			max: updatedDimConfig.auto_max
		}
        await sd.putDataZmq('/dimming', backendDimmingData);
		logToBackend(`Dimming update: ${logMsg}`, logLevels.Info);
	});

	res.send(JSON.stringify({
		success: true
	}));
});

app.put('/api/font', async (req, res) => {
	req.on('data', async function (data) {
        let fontConfig;
        await sd.getDataZmq('/font').then( data => { fontConfig = data; });

		let fontData = JSON.parse(data.toString());
		let logMsg = ` Original: ${fontConfig.style}`;
		logMsg += ` | Updated: ${fontData.name}`;

		fontConfig.style = fontData.number + 1;
        await sd.putDataZmq('/font', fontConfig);
		logToBackend(`Font update: ${logMsg}`, logLevels.Info);
	});

	res.send(JSON.stringify({
		success: true
	}));
});

app.get('/api/font', async (req, res) => {
    let fontConfig;
    await sd.getDataZmq('/font').then( data => { fontConfig = data; });
	let fontNumber = -1

	if (fontConfig) {
		let style = fontConfig.hasOwnProperty("style") ? fontConfig.style : false;
		if (style) {
			fontNumber = FontTypes[fontConfig["style"]] - 1;
		}
	} else {
		logToBackend(`GET /api/font: Could not get font config from FL5 backend! Is it running?`, logLevels.Error, false);
        sd.logToUI("Error retrieving font configuration", true, logLevels.Error);
	}

	res.send(JSON.stringify({
		"font_number": fontNumber
	}));
});

app.put('/api/frames', async (req, res) => {
	let success = false;

	req.on('data', async (data) => {
        let frameConfigBackend;
        await sd.getDataZmq('/frames').then( data => { frameConfigBackend = data; });
		let framesData = JSON.parse(data.toString())

		if (Object.hasOwn(framesData, "frames") || Object.hasOwn(framesData, "holdTime")) {
			let frameConfig = [];
			for (let idx = 0; idx < framesData.frames; ++idx) {
				frameConfig.push({ "number": idx + 1, "holdSeconds": framesData.holdTime });
			}

            await sd.putDataZmq('/frames', { "frames": frameConfig });

			let logMsg = "";
			if (frameConfigBackend) {
				logMsg += "\nOriginal:";
				for (fIdx = 0; fIdx < frameConfigBackend.frames.length; ++fIdx) {
					logMsg += ` [ F${frameConfigBackend.frames[fIdx].number} - hold sec: ${frameConfigBackend.frames[fIdx].holdSeconds} ]`
				}
			}
			logMsg += "\nUpdated:";
			for (fIdx = 0; fIdx < frameConfig.length; ++fIdx) {
				logMsg += ` [ F${frameConfig[fIdx].number} - hold sec: ${frameConfig[fIdx].holdSeconds} ]`
			}
			logToBackend(`Frame config update: ${logMsg}`, logLevels.Info);
			success = true;
		}

		if (!success) {
			logToBackend(`PUT /api/frames: Unable to successfully submit frames update`, logLevels.Error, false);
            sd.logToUI("Error submitting frame configuration", true, logLevels.Error);
		}
	});

	res.send(JSON.stringify({
		error: success
	}));
});

app.get('/api/frames', async (req, res) => {
    let frameConfig;
    await sd.getDataZmq('/frames').then( data => { frameConfig = data; });

	if (frameConfig && frameConfig.frames && frameConfig.frames.length > 0) {
		res.send(JSON.stringify({
			"frames": frameConfig.frames.length,
			"hold_time": frameConfig.frames[0]["holdSeconds"]
		}));
	} else {
		res.send(JSON.stringify({
			error: true
		}));
	}
});

app.get('/api/format', async (req, res) => {
    let formatData;
    await sd.getDataZmq('/format').then( data => { formatData = data; });

	res.send(JSON.stringify({
		tenths: formatData.tenths
	}));
});

app.put('/api/format', (req, res) => {
	req.on('data', async function (data) {
        let formatDataBackend;
        await sd.getDataZmq('/format').then( data => { formatDataBackend = data; });

		let dataJSON = JSON.parse(data.toString());
		let logMsg = ` Original: ${formatDataBackend.tenths}`;
		logMsg += ` | Updated: ${dataJSON.tenths}`;

		formatDataBackend.tenths = dataJSON.tenths;
        await sd.putDataZmq('/format', { tenths: dataJSON.tenths });
		logToBackend(`Format update: ${logMsg}`, logLevels.Info);
	});

	res.send(JSON.stringify({
		success: true
	}));
});

app.get('/api/cabAssyConf', async (req, res) => {
    let cabAssyConfigs = null;
    await sd.getDataZmq('/cabinet_assembly_configurations').then( data => { cabAssyConfigs = data; });

	if (cabAssyConfigs) {
        res.send(JSON.stringify(cabAssyConfigs.cabinetAssemblyConfigurations));
	} else {
		logToBackend(`GET /api/cabAssyConf: Could not find cabinet assembly configurations`, logLevels.Error, false);
        sd.logToUI("Error retrieving list of cabinet assembly configurations", true, logLevels.Error);
		res.send(JSON.stringify([]));
	}

});

app.get('/api/cashCreditMessages', async (req, res) => {
	if (!req.query.hwId) {
		logToBackend(`GET /api/cashCreditMessages: Missing query parameter hwId (hardware id) in request`, logLevels.Error, false);
	}

    let cabAssyConfigs = [];
	let cashCreditMessages = [];
	let hwIdGroups = [];
    await sd.getDataZmq('/cabinet_assembly_configurations').then( data => { cabAssyConfigs = data.cabinetAssemblyConfigurations; });
    await sd.getDataZmq('/cash_credit_messages').then( data => { cashCreditMessages = data; });
    await sd.getDataZmq('/hw_id_groups').then( data => { hwIdGroups = data; });

	const hwId = req.query.hwId;
	const length = req.query.length;
	let dispMsg = [];
	let errors = [];
	let warnings = [];

	if (hwId.startsWith('0x') && hwId.length >= 4) {
		let row = `0x0${hwId[3]}`;
		// Get the correct assembly configuration to match the panel width and height
		const cabAssyConf = cabAssyConfigs.filter((conf) => {
			return row == conf["hwId"] && length == conf["panels"];
		});
		// Get the correct ID group to match the pixel pitch
		const hwIdGroup = hwIdGroups["hwIdRowInfo"].filter((idGrp) => {
			return row == idGrp["hwId"];
		});
		if (cabAssyConf.length > 0 && hwIdGroup.length > 0) {
			const ccKey = `${cabAssyConf[0]["numPanelsHigh"]}hx${cabAssyConf[0]["numPanelsWide"]}w` // {panels wide}hx{panels wide}w (ex. 1hx2w)
			if (cashCreditMessages[ccKey]) {
				dispMsg = cashCreditMessages[ccKey].filter((msgConf) => {
					return msgConf["pixelPitchMm"] == hwIdGroup[0]["pixelPitchMm"];
				})[0];
				if (dispMsg && dispMsg.hasOwnProperty("messages") && dispMsg.messages.length > 0) {
					dispMsg = dispMsg.messages;
				} else {
					errors.push("msg_not_found"); // Could not find display message with pixel pitch of ${hwIdGroup[0]["pixelPitchMm"]} and ${ccKey} configuration
					dispMsg = [];
				}
			} else {
				errors.push("msg_missing"); // Could not find cash credit message in the dict of possible messages. Modules may be configured incorrectly
			}
		} else {
			if (cabAssyConf.length == 0) {
				warnings.push("assy_config");
			}
			if (hwIdGroup.length == 0) {
				warnings.push("hwid_grp");
			}
		}
	}

	if (errors.length > 0) {
		res.send(JSON.stringify({
			"errors": errors
		}));
	} else {
		res.send(JSON.stringify({
			"cash_credit_messages": dispMsg
		}));
	}
});

app.put('/api/updates', async (req, res) => {
	req.on('data', function (data) {
		let dataString = JSON.parse(data.toString());
		let operation = dataString.operation;
		let method = dataString.method;
		if (operation == "exec") {
            let version = dataString.version;
            let log_string = `Version ${version} obtained from ${method} source. Proceeding with update...`;
            // ############# Sanitize #############
            logToBackend(log_string, logLevels.Info, true);
			res.send(JSON.stringify({
				// Send a response to say that the request was acknowledged
				"status": log_string
			}));
			let stdout = execSync(`/usr/share/vega/ui/update_scripts/execUpdate.sh ${method}`).toString();
		} else if (method == "local") {
			let status = "none";
			let version = "UNKNOWN";
            try {
			    let stdout = execSync("sudo /usr/share/vega/ui/update_scripts/checkForLocalUpdate.sh", {"timeout" : 120000}).toString();
                
                if (stdout.indexOf("USB Not Found") > -1) {
                    status = "No USB";
                } else if (stdout.indexOf("Local Update Found") > -1) {
                    status = "local";
                }
                if (status == "local") {
                    stdout = execSync("dpkg -I /etc/vega/vega-software_*_arm64.deb").toString();
                    version = stdout.substring(stdout.indexOf("Version:"), stdout.indexOf("Installed-Size:")).split(' ')[1];
                    version = version.replace(/[\r\n]+/gm, "");
                }
            } catch (error) {
                status = "Timeout";
            }
			res.send(JSON.stringify({
				// Send a response if an update file is located or not
				"status": status,
				"version": version
			}));
		} else if (method == "remote") {
			let status = "none";
			let version = "UNKNOWN";
            try {
                let stdout = execSync("sudo /usr/share/vega/ui/update_scripts/checkForRemoteUpdate.sh", {"timeout" : 120000}).toString();

                if (stdout.indexOf("Offline") > -1) {
                    status = "Offline";
                } else if (stdout.indexOf("Remote Update Found") > -1) {
                    status = "remote";
                }
                if (status == "remote") {
                    stdout = execSync("dpkg -I /etc/vega/firmware.deb").toString();
                    version = stdout.substring(stdout.indexOf("Version:"), stdout.indexOf("Installed-Size:")).split(' ')[1];
                    version = version.replace(/[\r\n]+/gm, "");
                }
            }
            catch (error) {
                status = "Timeout";
            }
			res.send(JSON.stringify({
				// Send a response if an update file is located or not
				"status": status,
				"version": version
			}));
		}
	});
});

app.put('/api/runTest', async (req, res) => {
	req.on('data', async function (data) {
		let testObj = JSON.parse(data.toString());
        await sd.putDataZmq('/run_test', { "command": testObj.type, "value": testObj.subtype, "description": testObj.name });
        if (testObj.subtype == 0) {
            logToBackend(`Stopping any running tests and reverting to normal functionality`, logLevels.Info, false);
        } else {
            logToBackend(`Running test: ${testObj.name}`, logLevels.Info);
        }
		res.send(JSON.stringify({
			"status": "sent"
		}));
	});
});

app.put('/api/runDispayTPat', async (req, res) => {
	req.on('data', async function (data) {
		let msg = JSON.parse(data.toString());
        await sd.putDataZmq('/run_display_tpat', msg);
		logToBackend(`Running test pattern: ${msg.test_pattern}`, logLevels.Info);
		res.send(JSON.stringify({
			"status": "sent"
		}));
	});
});

app.post('/api/bootload', async (req, res) => {
	req.on('data', async function () {
        await sd.getDataZmq('/bootload_displays');
		logToBackend(`Bootloading Displays`, logLevels.Info);
		res.send(JSON.stringify({
			"status": "sent"
		}));
	});
});

app.post('/api/reboot', async (req, res) => {
	req.on('data', async function () {
        await sd.getDataZmq('/reset_displays');
		logToBackend(`Rebooting Displays`, logLevels.Info);
		res.send(JSON.stringify({
			"status": "sent"
		}));
	});
});

app.get('/api/clientRadioConfig', async (req, res) => {
    logToBackend(`Retrieving radio configuration`, logLevels.Info);
    await sd.getDataZmq('/client_radio_config');
    res.send(JSON.stringify([]));
});

app.get('/api/networkPriceUpdateStatus', async (req, res) => {
    logToBackend(`Retrieving network price update status`, logLevels.Info);
    await sd.getDataZmq('/network_price_update_status');
    res.send(JSON.stringify([]));
});

app.post('/api/clientRadioConfig', async (req, res) => {
	req.on('data', async (data) => {
		let uiRadioConfig = JSON.parse(data.toString());
		logToBackend(`Applying radio configuration: ${JSON.stringify(uiRadioConfig, null, "")}`, logLevels.Info);
        await sd.putDataZmq('/client_radio_config', uiRadioConfig);

        res.send(JSON.stringify([]));
	});
});

app.get('/api/logContent', async (req, res) => {
	const TotalLines = 1000; // Return the last 1000 lines in the log file

	let logFile = ""
	let currentMtime = 0;
    let fileList = fs.readdirSync(sd.logDir, { withFileTypes: true })
        .filter(dirent => dirent.isFile())
        .map(dirent => dirent.name);
	for (let idx = 0; idx < fileList.length; ++idx) {
		filePath = path.join(sd.logDir, fileList[idx]);
		const stats = fs.statSync(filePath);
		if (stats.mtime > currentMtime) {
			currentMtime = stats.mtime;
			logFile = filePath;
		}
	}

	if (logFile == "") {
		res.send(JSON.stringify({
			success: true,
			content: ""
		}));
		return;
	} else {
        // Shouldn't need to do any sanitizing here these aren't entered by the user
        exec(`tail -n ${TotalLines} ${logFile}`, function(err, stdout, _){
            if(err){
                sd.logToUI("Error retrieving log content");
                res.status(500).send({"Error retrieving log content": err.toString()});
            }
            else{
                res.send(JSON.stringify({
                    success: true,
                    content: String(stdout)
                }));
            }
        });
    }
});

app.get('/api/localNetIP', async (req, res) => {
	let deviceStr = "eth";
	let localIP = "";

	const networkInterfaces = os.networkInterfaces();
	for (const name of Object.keys(networkInterfaces)) {
		if (name.includes(deviceStr)) {
			for (const net of networkInterfaces[name]) {
				if (net.family == "IPv4" && !net.internal) {
					localIP = net.address;
				};
			}
		}
	}

	res.send(JSON.stringify({
		success: true,
		ip: localIP
	}));
});

app.get('/api/metaData', async (req, res) => {
	let metaData = sd.getMetaData();
	res.send(JSON.stringify({
		modFWVersion: metaData.mod_fw_version,
		fontFileVersion: metaData.font_file_version,
		vegaSWVersion: metaData.vega_sw_version
	}));
});

app.put('/api/stopRetries', async (req, res) => {
    await sd.putDataZmq('/stop_retries', { code: 0, description: "" });
	logToBackend(`Stopping retries`, logLevels.Info);
	res.send();
});

app.get('/api/appSettings', async (req, res) => {
    let appSettings;
	await sd.getDataZmq('/app_settings').then( data => { appSettings = data; });

	if (!appSettings || appSettings == {}) {
		console.log("Could not get app settings")
		res.send([]);
		return;
	}

	res.send(appSettings);
});

app.put('/api/appSettings', async (req, res) => {
	req.on('data', async function (data) {
		let appSettings = JSON.parse(data.toString());
		let backendAppSettings = {};
        await sd.getDataZmq('/app_settings').then( apData => { backendAppSettings = apData; });

        if (!backendAppSettings || backendAppSettings == {}) {
            console.log("Could not get app settings")
            res.send(JSON.stringify({
                success: false,
            }));
            return;
        }

        if (backendAppSettings.hasOwnProperty("ertdSupport")) {
            backendAppSettings["ertdSupport"] = appSettings.hasOwnProperty("ertd_support") ? appSettings.ertd_support : backendAppSettings.ertdSupport;
        }
        if (backendAppSettings.hasOwnProperty("ertdBroadcast")) {
            backendAppSettings["ertdBroadcast"] = appSettings.hasOwnProperty("ertd_broadcast") ? appSettings.ertd_broadcast : backendAppSettings.ertdBroadcast;
        }

        await sd.putDataZmq('/app_settings', backendAppSettings);
		res.send(JSON.stringify({
			success: true,
		}));
	});
});

app.get('/api/ertdConfig', async (req, res) => {
    let ertdConfig;
	await sd.getDataZmq('/ertd_configuration').then( data => { ertdConfig = data; });

	if (!ertdConfig || ertdConfig == {}) {
		console.log("Could not get app settings")
		res.send([]);
		return;
	}

	res.send(ertdConfig);
});

app.put('/api/ertdConfig', async (req, res) => {
	req.on('data', async function (data) {
		let ertdConfig = JSON.parse(data.toString());

        await sd.putDataZmq('/ertd_configuration', ertdConfig);
		res.send(JSON.stringify({
			success: true,
		}));
	});
});


app.put('/api/enableSSH', async (req, res) => {
    req.on('data', async (data) => {
		let sshConfig = JSON.parse(data.toString());
        let result = "SSH mode not changed";
        if (sshConfig.sshMode)
        {
            logToBackend(`Enabling SSH...`, logLevels.Info);
            result = execSync("systemctl enable ssh; systemctl start ssh").toString();
        }
        else
        {
            logToBackend(`Disabling SSH and terminating active connections...`, logLevels.Info);
            result = execSync("systemctl stop ssh; systemctl disable ssh; killall sshd || true").toString();
        }

        res.send(JSON.stringify({
            success: true,
            output: result
        }));
	});
});

app.get('/api/enableSSH', async (req, res) => {
	let result = execSync("systemctl is-enabled ssh || true").toString();
    let sshStatus = result.indexOf("enabled") > -1;

	res.send(JSON.stringify({
		sshMode: sshStatus
	}));
});

app.get('/api/macAddresses', async (req, res) => {
	let eth0_mac = execSync("ip addr show eth0 | grep -i ether | awk '{print $2}'").toString();
	let wlan0_mac = execSync("ip addr show wlan0 | grep -i ether | awk '{print $2}'").toString();

	res.send(JSON.stringify({
		success: true,
		mac: { 'eth0': eth0_mac, 'wlan0': wlan0_mac }
	}));
});

app.get('/api/apConfig', async (req, res) => {
	const hostapdFile = sd.hostapdFile
	const bckHostapdFile = sd.bckHostapdFile
	const bckVegaNetwrkConf = sd.bckVegaNetwrkConf
	const vegaNetwrkConfFile = sd.vegaNetwrkConf
    let ap_open = false;
    let bck_hostapd_pass = "";
    let bck_hostapd_ssid = "";
    let bck_enable_ssid = false;
    let bck_enable_ap = false;

	await sd.getDataZmq('/ap_status').then( data => { ap_open = data.status; });

	let configData = {
		"ssid": "",
		"enable_ssid": false,
		"enable_ap": false,
		"password": "",
        "ap_open": false
	}

	const ntConfData = fs.readFileSync(vegaNetwrkConfFile, { encoding: 'utf8', flag: 'r' });
	split_nt_conf_data = ntConfData.split("\n");
	for (let idx = 0; idx < split_nt_conf_data.length; ++idx) {
		let line = split_nt_conf_data[idx];

		if (line.startsWith("enable_ap")) {
			const enable_ap_start = line.indexOf('=');
			if (enable_ap_start >= 0) {
				const value = line.slice(enable_ap_start + 1);
				configData.enable_ap = parseInt(value) == 1;
			}
		}
	}

    if (fs.existsSync(bckVegaNetwrkConf)) {
        const bck_net_cfg_data = fs.readFileSync(bckVegaNetwrkConf, { encoding: 'utf8', flag: 'r' });
        split_bck_net_cfg_data = bck_net_cfg_data.split("\n");
        for (let idx = 0; idx < split_bck_net_cfg_data.length; ++idx) {
            let line = split_bck_net_cfg_data[idx];
            if (line.startsWith("enable_ap")) {
                const enable_ap_start = line.indexOf('=');
                if (enable_ap_start >= 0) {
                    const value = line.slice(enable_ap_start + 1);
                    bck_enable_ap = parseInt(value) == 1;
                }
            }
        }
    }

    if (fs.existsSync(bckHostapdFile)) {
        const bck_hostapd_data = fs.readFileSync(bckHostapdFile, { encoding: 'utf8', flag: 'r' });
        split_bck_hostapd_data = bck_hostapd_data.split("\n");
        for (let idx = 0; idx < split_bck_hostapd_data.length; ++idx) {
            let line = split_bck_hostapd_data[idx];
            if (line.startsWith("wpa_passphrase")) {
                const pass_start = line.indexOf('=');
                if (pass_start >= 0) {
                    bck_hostapd_pass = line.slice(pass_start + 1);
                }
            }
            if (line.startsWith("ssid")) {
                const ssid_start = line.indexOf('=');
                if (ssid_start >= 0) {
                    bck_hostapd_ssid = line.slice(ssid_start + 1);
                }
            }
            if (line.startsWith("ignore_broadcast_ssid")) {
                const ssid_enabled_start = line.indexOf('=');
                if (ssid_enabled_start >= 0) {
                    const ign_value = line.slice(ssid_enabled_start + 1);
                    bck_enable_ssid = parseInt(ign_value) == 0;
                }
            }
        }
    }

	const hostapd_data = fs.readFileSync(hostapdFile, { encoding: 'utf8', flag: 'r' });
	split_hostapd_data = hostapd_data.split("\n");
	for (let idx = 0; idx < split_hostapd_data.length; ++idx) {
		let line = split_hostapd_data[idx];

		if (line.startsWith("ssid")) {
			const ssid_start = line.indexOf('=');
			if (ssid_start >= 0) {
				configData.ssid = line.slice(ssid_start + 1);
			}
		}

		if (line.startsWith("ignore_broadcast_ssid")) {
			const ssid_enabled_start = line.indexOf('=');
			if (ssid_enabled_start >= 0) {
				const ign_value = line.slice(ssid_enabled_start + 1);
				configData.enable_ssid = parseInt(ign_value) == 0; // If ignore_broadcast_ssid is 0 the ssid will be broadcast
			}
		}

		if (line.startsWith("wpa_passphrase")) {
			const wpa_passphrase_start = line.indexOf('=');
			if (wpa_passphrase_start >= 0) {
				configData.password = line.slice(wpa_passphrase_start + 1);
			}
		}
	}

    configData.ap_open = ap_open && configData.password == "";
    if (ap_open && configData.password == "") {
        configData.password = bck_hostapd_pass;
        configData.ssid = bck_hostapd_ssid;
        configData.enable_ssid = bck_enable_ssid;
        configData.enable_ap = bck_enable_ap;
    }

	res.send(JSON.stringify({
		success: true,
		...configData
	}));
});

function get_hostapd_pass_ssid() {
    const hostapdFile = sd.hostapdFile
    const vegaNetwrkConfFile = sd.vegaNetwrkConf
    let hostapd_pass = "";
    let hostapd_ssid = "";
    let enable_ssid = 0;
    let enable_ap = 0;

    if (fs.existsSync(hostapdFile)) {
        const hostapd_data = fs.readFileSync(hostapdFile, { encoding: 'utf8', flag: 'r' });
        split_hostapd_data = hostapd_data.split("\n");
        for (let idx = 0; idx < split_hostapd_data.length; ++idx) {
            let line = split_hostapd_data[idx];
            if (line.startsWith("wpa_passphrase")) {
                const pass_start = line.indexOf('=');
                if (pass_start >= 0) {
                    hostapd_pass = line.slice(pass_start + 1);
                }
            }
            if (line.startsWith("ssid")) {
                const ssid_start = line.indexOf('=');
                if (ssid_start >= 0) {
                    hostapd_ssid = line.slice(ssid_start + 1);
                }
            }
            if (line.startsWith("ignore_broadcast_ssid")) {
                const ssid_enable_start = line.indexOf('=');
                if (ssid_enable_start >= 0) {
                    enable_ssid = line.slice(ssid_enable_start + 1);
                }
            }
        }
    };

	const ntConfData = fs.readFileSync(vegaNetwrkConfFile, { encoding: 'utf8', flag: 'r' });
	split_nt_conf_data = ntConfData.split("\n");
	for (let idx = 0; idx < split_nt_conf_data.length; ++idx) {
		let line = split_nt_conf_data[idx];

		if (line.startsWith("enable_ap")) {
			const enable_ap_start = line.indexOf('=');
			if (enable_ap_start >= 0) {
				const value = line.slice(enable_ap_start + 1);
				enable_ap = parseInt(value) == 1;
			}
		}
	}

    return {
        "password": hostapd_pass,
        "ssid": hostapd_ssid,
        "enable_ssid":  enable_ssid,
        "enable_ap":  enable_ap
    };
}

function remove_config_lines(hostapdFile) {
	return new Promise((resolve) => {
		hostapd_content_out = "";

		const file = readline.createInterface({
			input: fs.createReadStream(hostapdFile),
			output: process.stdout,
			terminal: false
		});

		// Go through the file line by line remove the things being configured from the output file
		file.on('line', (line) => {
			let append_line = line;
			if (
				!line.startsWith("ssid") &&
				!line.startsWith("wpa_psk") &&
				!line.startsWith("ignore_broadcast_ssid") &&
				!line.startsWith("wpa_passphrase") &&
				!line.startsWith("wpa_key_mgmt") &&
				!line.startsWith("wpa_pairwise") &&
				!line.startsWith("rsn_pairwise") &&
				!line.startsWith("wpa")
			) {
				if (hostapd_content_out == "") {
					hostapd_content_out = hostapd_content_out.concat(append_line);
				} else {
					hostapd_content_out = hostapd_content_out.concat('\n', append_line);
				}
			}
		});

		file.on('close', () => {
			resolve(hostapd_content_out);
		});

		file.on('error', () => {
			resolve(false);
		});
	});
}

app.put('/api/apConfig', async (req, res) => {
	req.on('data', function (data) {
		const hostapdFile = sd.hostapdFile
		const vegaNetwrkConfFile = sd.vegaNetwrkConf

		let apConfig = JSON.parse(data.toString());
        const conf_hostapd_settings = get_hostapd_pass_ssid();
		const ssid = apConfig.hasOwnProperty("ssid") ? apConfig.ssid : conf_hostapd_settings.ssid;
		const password = apConfig.hasOwnProperty("password") ? apConfig.password : conf_hostapd_settings.password;
        const enable_ssid = apConfig.hasOwnProperty("enable_ssid") ? apConfig.enable_ssid : conf_hostapd_settings.enable_ssid;
        const enable_ap = apConfig.hasOwnProperty("enable_ap") ? apConfig.enable_ap : conf_hostapd_settings.enable_ap;

		logToBackend(`Applying access point configuration: ${JSON.stringify(apConfig, null, "")}`, logLevels.Info);
        // Unused - If we decide to enable this to hash our passwords this will need sanitization
		//const wpa_psk = execSync(`wpa_passphrase ${ssid} ${password} | awk '{$1=$1};1' | grep -P '^psk=' | awk -F '=' '{print $2}'`).toString();

		remove_config_lines(hostapdFile).then(async (hostapd_content_out) => {
			if (hostapd_content_out) {
				const apdWS = fs.createWriteStream(hostapdFile, { flags: 'w+' });
				sd.logToUI(`Setting SSID to: ${ssid}`, true, logLevels.Info);
				apdWS.write(hostapd_content_out);
				apdWS.write(`\nssid=${ssid}\n`);
				apdWS.write(`wpa_passphrase=${password}\n`);

				apdWS.write(`wpa=2\n`);
				apdWS.write(`wpa_key_mgmt=WPA-PSK\n`);
				apdWS.write(`wpa_pairwise=TKIP\n`);
				apdWS.write(`rsn_pairwise=CCMP\n`);

				// apdWS.write(`wpa_psk=${wpa_psk}\n`); Uncomment this and comment the next line to use the encrypted password
				apdWS.write(`ignore_broadcast_ssid=${enable_ssid ? 0 : 2}`);
				enable_ssid ? logToBackend("Enabling SSID broadcast", logLevels.Info) : logToBackend("Disabling SSID broadcast", logLevels.Info);
				// Send empty SSID in beacons and ignore probe request frames that do not
				// specify full SSID, i.e., require stations to know SSID.
				// default: disabled (0)
				// 1 = send empty (length=0) SSID in beacon and ignore probe request for
				//     broadcast SSID
				// 2 = clear SSID (ASCII 0), but keep the original length (this may be required
				//     with some clients that do not support empty SSID) and ignore probe
				//     requests for broadcast SSID
				apdWS.end();
				await new Promise(resolve => setTimeout(resolve, 500)); // Wait a bit for the stream to close

				const ntwrkWS = fs.createWriteStream(vegaNetwrkConfFile, { flags: 'w' }); // No w+ here, only one variable at the time of writing so no appending
				enable_ap ? logToBackend("Enabling access point", logLevels.Info) : logToBackend("Disabling access point", logLevels.Info);
				ntwrkWS.write(`enable_ap=${enable_ap ? 1 : 0}`);
				ntwrkWS.end();

				logToBackend("Restarting networking services, this will take some time. Access point connection will be lost temporarily. Refresh the browser until the UI is available again.", logLevels.Info);
				await new Promise(resolve => setTimeout(resolve, 500));
				execSync("sudo /usr/bin/vega-netstop");
				await new Promise(resolve => setTimeout(resolve, 500));
				execSync("sudo /usr/bin/vega-netstart");
				await new Promise(resolve => setTimeout(resolve, 500));
				logToBackend("Access point settings applied", logLevels.Info);

                await sd.putDataZmq('/ap_status', { "command": "reset_ap_open_timer" });

			} else {
				logToBackend(`PUT /api/apConfig - hostapd config file empty!: ${JSON.stringify(error, null, "")}`, logLevels.Error, false);
                sd.logToUI("hostapd config file on device is empty!", true, logLevels.Error);
			}

		});
	});

    res.send(JSON.stringify({
        "success": true
    }));
});

function configure_static_ip(config) {
	return new Promise((resolve) => {
		const dhcpFile = sd.dhcpFile;
		static_section = false;
		dhcp_content_out = "";

		const file = readline.createInterface({
			input: fs.createReadStream(dhcpFile),
			output: process.stdout,
			terminal: false
		});

		// Go through the file line by line, exclude the static section from the output content
		file.on('line', (line) => {
			if (line.includes("STATIC_IP_START")) {
				static_section = true;
			}

			if (!static_section) {
				if (dhcp_content_out == "") {
					dhcp_content_out = dhcp_content_out.concat(line);
				} else {
					dhcp_content_out = dhcp_content_out.concat('\n', line);
				}
			}

			if (line.includes("STATIC_IP_END")) {
				static_section = false;
			}
		});

		file.on('close', () => {
			const writeStream = fs.createWriteStream(dhcpFile, { flags: 'w+' });
			writeStream.write(dhcp_content_out);

			if (config.use_static_ip) {
				writeStream.write('\n#__STATIC_IP_START__\n');
				writeStream.write('interface eth0\n');
				writeStream.write(`static ip_address=${config.static_ip}/${config.subnet_mask_slash}\n`);
				writeStream.write(`static routers=${config.default_gateway}\n`);
				writeStream.write(`static domain_name_servers=${config.dns_address}\n`);
				writeStream.write('#__STATIC_IP_END__\n');
			}
			writeStream.end();
			sd.logToUI("Reconfigured DHCPCD");
			resolve(true);
		});

		file.on('error', () => {
			resolve(false);
		});
	});
}

app.put('/api/ipConfig', async (req, res) => {
	req.on('data', function (data) {
		let ipConfig = JSON.parse(data.toString());

		logToBackend(`Applying new IP configuration: ${JSON.stringify(ipConfig, null, "")}`, logLevels.Info);
		configure_static_ip(ipConfig).then((success) => {
			if (success) {
				sd.logToUI("Updating Network Settings\nNavigate to new IP");
				exec('sudo node update_ip.js');
				execute('sudo node update_ip.js', sd.logToUI);
			} else {
				logToBackend(`PUT /api/ipConfig: Unable to successfully submit ipConfig`, logLevels.Error, false);
                sd.logToUI("Failed to apply new IP settings", true, logLevels.Error);
			}
			res.send(JSON.stringify({
				"success": success
			}));

		});

	});
});

app.get('/api/ipConfig', async (req, res) => {
	const dhcpFile = sd.dhcpFile
	let configData = {
		"use_static_ip": false,
		"static_ip": "",
		"subnet_mask_slash": 0,
		"dns_address": "",
		"default_gateway": "",
	}

	in_static_section = false;
	found_static_section = false;

	const file = readline.createInterface({
		input: fs.createReadStream(dhcpFile),
		output: process.stdout,
		terminal: false
	});

	file.on('line', (line) => {
		if (!in_static_section && line.includes("STATIC_IP_START")) {
			in_static_section = true;
			found_static_section = true;
		} else if (in_static_section && line.includes("STATIC_IP_END")) {
			in_static_section = false;
		}

		// Extract the values associated with the configuration of the static ip settings
		if (in_static_section) {
			if (line.includes("ip_address")) {
				const ip_start = line.indexOf('=');
				const mask_start = line.indexOf('/');
				if (ip_start >= 0 && mask_start >= 0) {
					configData.static_ip = line.slice(ip_start + 1, mask_start);
					configData.subnet_mask_slash = parseInt(line.slice(mask_start + 1));
				} else {
					configData.static_ip = "";
					configData.subnet_mask_slash = 0;
				}

			} else if (line.includes("routers")) {
				const gateway_start = line.indexOf('=');
				configData.default_gateway = line.slice(gateway_start + 1);
			} else if (line.includes("domain_name_servers")) {
				const dns_address_start = line.indexOf('=');
				configData.dns_address = line.slice(dns_address_start + 1);
			}
		}
	});

	file.on('close', () => {
		configData.use_static_ip = found_static_section;
		res.send(JSON.stringify({
			success: true,
			config: configData
		}));
	});
});

app.put('/api/updateModStatus', async (req, res) => {
	req.on('data', async function (data) {
        await sd.putDataZmq('/mod_status', { "command": data.toString() });
		res.send(JSON.stringify({
			success: true,
		}));
	});
});


app.get('/api/getRSSIData', async (req, res) => {
	res.send(JSON.stringify({
		"data": sd.getRSSIData()
	}));
});

app.put('/api/modCalibration', async (req, res) => {
	req.on('data', async function (data) {
        await sd.putDataZmq('/calibrate_mod', JSON.parse(data.toString()));
		res.send({
			"success": true
		});
	});
});

app.get('/api/uploadDefaultFontFile', async (req, res) => {
    sd.logToUI("uploadDefaultFontFile", true, logLevels.Verbose, false);
    await sd.putDataZmq('/upload_font_file', { description: sd.fontFile });

    res.status(200).json({ message: 'File sent successfully' });
});

app.post('/api/uploadFontFile', upload.single('file'), async (req, res) => {
    if (req.file) {
        await sd.putDataZmq('/upload_font_file', { description: path.join(uploadPath, req.file.originalname) });
        res.status(200).json({ message: 'File uploaded successfully' });
    } else {
        res.status(400).json({ error: 'File upload failed' });
    }
});

app.get('/api/licensing/angular', (req, res) => {

    try
    {
        res.send({text : fs.readFileSync(__dirname + '/licensing/ui/angular/LICENSES.txt', 'utf8')});
    }
    catch (err)
    {
        console.error(err);
        res.send({text : `${err}`});
    }
});

app.get('/api/licensing/node', (req, res) => {

    try
    {
        res.send({text : fs.readFileSync(__dirname + '/licensing/ui/nodejs/LICENSES.txt', 'utf8')});
    }
    catch (err)
    {
        console.error(err);
        res.send({text : `${err}`});
    }
});

app.get('/api/licensing/backend', (req, res) => {

    try
    {
        res.send({text : fs.readFileSync(__dirname + '/licensing/backend/LICENSES.txt', 'utf8')});
    }
    catch (err)
    {
        console.error(err);
        res.send({text : `${err}`});
    }
});

app.get('/api/licensing/backend_other', (req, res) => {

    try
    {
        res.send({text : fs.readFileSync(__dirname + '/licensing/backend/licensing.html', 'utf8')});
    }
    catch (err)
    {
        console.error(err);
        res.send({text : `${err}`});
    }
});

function getWiFiStatus(wifiSSID) {
    let wifiStatus = 0;

	let connectionJson = getIpAddresses();
	let wlan0Active = false;

	for (connection of connectionJson) {
		if (connection.name == "wlan0") {
			wlan0Active = true;
		}
	}

	if (wlan0Active) {
		wifiStatus = 1;
	}
	else if (wifiSSID == "") {
		wifiStatus = 0;
	}
	else {
		wifiStatus = -1;
	}

    return wifiStatus;
}

function getIpAddresses() {
	let interfaces = require('os').networkInterfaces();
	let connectionArray = new Array();
	for (const name of Object.keys(interfaces)) {
		for (const net of interfaces[name]) {
			// Skip over non-IPv4 and internal (i.e. 127.0.0.1) addresses
			if (net.family === 'IPv4' && !net.internal && name != "uap0") {
				let connection = new Object();
				connection.name = name;
				connection.address = net.address;
				connectionArray.push(connection);
			}
		}
	}
	return connectionArray;
}

function generatePSK(ssid, passphrase) {
  // Key derivation using PBKDF2 with HMAC-SHA1, 4096 iterations, and 32-byte output
  const psk = crypto.pbkdf2Sync(passphrase, ssid, 4096, 32, 'sha1');
  return psk.toString('hex');  // Convert the buffer to a hexadecimal string
}

let network_failures = 0;
let monitor_network_connection;
// Handle network update wifi changes
function update_wpa_configuration(network, password) {
	let data = "ctrl_interface=DIR=/var/run/wpa_supplicant GROUP=netdev\nupdate_config=1\ncountry=US\n\n";
	if (network != "") {
		if (password != "") {
            // Calling this function is a substitute for running wpa_passphrase. By generating the psk via this function we don't need to run any user
            // supplied inputs from the request on the device.
            const psk = generatePSK(network,  password);
            data = data.concat(`network={\n\tssid="${network}"\n\tpsk=${psk}\n}`);
		}
		else {
			// If there is no password, create a network config without one
			data += "network={\n\tssid=\"" + network + "\"\n\tkey_mgmt=NONE\n}";
		}
	}

	execute('rm /etc/wpa_supplicant/wpa_supplicant.conf', (_) => {
		fs.writeFile('/etc/wpa_supplicant/wpa_supplicant.conf', data, err => {
			if (err) {
				console.log('Error writing file', err);
			}
			else {
				console.log('Successfully wrote wpa_supplicant file');
				execute('sudo systemctl daemon-reload', (_) => {
					console.log("daemon-reloaded");
					execute('sudo systemctl restart dhcpcd', (_) => {
                        console.log("restarted dhcpcd");
                        // Check the status of the wifi every 30 seconds and clear the wpa supplicant file if necessary
                        clearInterval(monitor_network_connection);
                        monitor_network_connection = setInterval(() => {
                            let wifiSSID;
                            let wifiPassword;

                            sd.getDataZmq('/wifi').then(data => {
                                wifiSSID = data.ssid;
                                wifiPassword = data.pass;

                                let num = getWiFiStatus(wifiSSID);
                                if (num == -1) {
                                    if (network_failures > 5) {
                                        sd.logToUI("Cannot establish good connection, clearing wifi configuration...", true, logLevels.Warning, false);
                                        clearInterval(monitor_network_connection);
                                        network_failures = 0;
                                        update_wpa_configuration("", "");
                                        sd.putDataZmq('/wifi', { ssid: "", pass: "" });
                                    } else {
                                        network_failures++;
                                    }
                                } else {
                                    clearInterval(monitor_network_connection);
                                    network_failures = 0;
                                }
                            });
                        }, 5000);
                    });
				});
			}
		});
	});

    return true;
}

// Execute terminal command 
function execute(command, callback) {
	exec(command, (error, stdout, stderr) => {
		if (error) {
			console.log("error: " + error);
		}
		callback(stdout);
	});
}


// #### This need to be at the very end of the file ####
app.get('/*', (req, res) => {
	res.sendFile(path.join(angularPath, 'index.html'));
});

