const fs = require('fs');
const zmq = require("zeromq");
const protobuf = require("protobufjs");
const process = require('process');
const path = require('path');
const Mutex = require('async-mutex').Mutex;

const { console_log_error, console_log_magenta, console_log_ok, console_log_cyan, console_log_blue, console_log_warn, logLevels } = require('./server_util');
const logDir = '/var/log/vega';
const configDir = "/etc/vega/config/fl5";
const protoDir = "/var/lib/vega/fl5";
const netDir = '/sys/class/net'
const dhcpFile = "/etc/dhcpcd.conf";
const versionFile = "/etc/vega/config/fl5/version.txt"
const fudFile = "/etc/vega/config/fl5/heracles_bootload.fud"
const fontFile = "/etc/vega/config/fl5/FL5xSPI.fud"
const hostapdFile = "/etc/hostapd/hostapd.conf";
const bckHostapdFile = "/etc/hostapd/hostapd.conf.bck";
const bckVegaNetwrkConf = "/etc/vega/config/fl5/vega-networking.config.bck";
const vegaNetwrkConf = "/etc/vega/config/fl5/vega-networking.config";

// Global variables
let Message;
let vegaRoot;
let networkRoot;
let Prices;
let FL5Products;
let POSDataMsg;
let TestUpdateMsg;
let Err;
let AddrStatus;
let subSocket;
let wss;
let wifiNetworks = [];
let rssiData = {};
let pingData = {
    response: false
};
let metaData = {
    // TODO: Find a way to pull these procedurally from the files, else find the best place to hard-code those values (here is probably preferred)
    vega_sw_version: "UNKNOWN", // Version of the Fuelink5 software
    mod_fw_version: "UNKNOWN", // Version of the Heracles firmware
    font_file_version: "UNKNOWN" // Version of the font file
};
let reqSocket = new zmq.Request({receiveTimeout: 10000});
const mutex = new Mutex();

// Module Exports
module.exports.putDataZmq = putDataZmq;
module.exports.getDataZmq = getDataZmq;
module.exports.getMetaData = getMetaData;
module.exports.getPingData = getPingData;
module.exports.setPingData = setPingData;
module.exports.getWifiNetworks = getWifiNetworks;
module.exports.setWifiNetworks = setWifiNetworks;
module.exports.connectToBackend = connectToBackend;
module.exports.getVersionInfo = getVersionInfo;
module.exports.getRSSIData = getRSSIData;
module.exports.logToUI = logToUI;
module.exports.logDir = logDir;
module.exports.configDir = configDir;
module.exports.protoDir = protoDir;
module.exports.netDir = netDir;
module.exports.dhcpFile = dhcpFile;
module.exports.hostapdFile = hostapdFile;
module.exports.fontFile = fontFile;
module.exports.bckHostapdFile = bckHostapdFile;
module.exports.bckVegaNetwrkConf = bckVegaNetwrkConf;
module.exports.vegaNetwrkConf = vegaNetwrkConf;
module.exports.reqSocket = reqSocket;

protobuf.load(path.join(__dirname, 'Message.proto')).then((root) => {
    Message = root.lookupType('kzmq.Message');
    let msg = `Server started with version v${metaData.vega_sw_version}`;
    let level = logLevels.Info;
    logToUI(msg, true, level, true);
    putDataZmq(
        '/log_to_file',
        { 
            "value": level,
            "description": "NodeJS - " +  msg
        }
    );
});

async function connectToBackend(root_dir, wss_in) {
	wss = wss_in

	vegaRoot = await protobuf.load(path.join(root_dir, 'Vega.proto'));
	networkRoot = await protobuf.load(path.join(root_dir, 'Network.proto'));
	Prices = vegaRoot.lookupType('vega.Prices');
	FL5Products = vegaRoot.lookupType('vega.FL5Products');
	Err = vegaRoot.lookupType('vega.Err');
	AddrStatus = networkRoot.lookupType('vega.AddressingStatus');
	POSDataMsg = vegaRoot.lookupType('vega.POSDataMsg');
	TestUpdateMsg = vegaRoot.lookupType('vega.TestUpdateMsg');
	ClientStatusMsg = vegaRoot.lookupType('vega.ClientStatusMsg');
	ClientRadioConfigMsg = vegaRoot.lookupType('vega.ClientRadioConfigMsg');
	BaseMsg = vegaRoot.lookupType('vega.BaseMsg');
	Display = networkRoot.lookupType('vega.Display');
	NetworkStatusReport = networkRoot.lookupType('vega.NetworkStatusReport');
	AppStates = vegaRoot.lookupType('vega.AppStates');

	await reqSocket.connect("ipc:///run/vega/fl5/zmq.sock");

	subSocket = new zmq.Subscriber();
	subSocket.connect("ipc:///var/run/vega/nodejs/sub.sock");
	subSocket.subscribe("/event");
    for await (const [_, data] of subSocket) {
		const message = Message.decode(data);

		if (message.put.head.path == "/event/fl5_products") {
			const fl5_products = FL5Products.decode(message.put.body.bin);
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "price-change", "message": fl5_products }));
			})
		} else if (message.put.head.path == "/event/prices") {
			const prices = Prices.decode(message.put.body.bin);
			//console.log( "message:", message, "payload:", prices );
            let fl3_products = [];
			for (let i = 0; i < 5; ++i) {
				fl3_products[i].id = prices.prices[i].productId;
			}
            await putDataZmq('/fl3_products', { "products": fl3_products });
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "price-change", "message": fl3_products }));
			})
		} else if (message.put.head.path == "/event/priceChangeStatus") {
			const status = Err.decode(message.put.body.bin);
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "price-change-status", "message": status }));
			})
		} else if (message.put.head.path == "/event/addressingStatus") {
			const status = AddrStatus.decode(message.put.body.bin);
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "detect-displays", "message": status }));
			})

			if (status.statusCode == 3) // 3 is the code for finished addressing of the network
			{
                logToUI(`Addressing Finished!  Linear Addresses: ${status.linearAddressAssignments}  Assembly Addresses:${status.assemblyAddressAssignments}`, true, logLevels.Info, false);
			}
			else {
                logToUI(`LinearAddresses: ${status.linearAddressAssignments}  Assembly Addresses:${status.assemblyAddressAssignments}`, true, logLevels.Verbose, false);
			}
		} else if (message.put.head.path == "/event/posData") {
            const defaultPosData = {
                "data": 0,
                "transmit": false
            }

			const posData = POSDataMsg.decode(message.put.body.bin);
            if (posData.hasOwnProperty("transmit")) {
                defaultPosData.transmit = posData.transmit;
            }
            if (posData.hasOwnProperty("data")) {
                defaultPosData.data = posData.data;
            }

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "pos-monitor", "message": defaultPosData }));
			})
		} else if (message.put.head.path == "/event/configStatus") {
			const status = Err.decode(message.put.body.bin);
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "config-displays", "message": status }));
			});
		} else if (message.put.head.path == "/event/testUpdate") {
			const testUpdateMsg = TestUpdateMsg.decode(message.put.body.bin);

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "test-update", "message": testUpdateMsg }));
			})
		} else if (message.put.head.path == "/event/clientStatusUpdate") {
			const clientStatusMsg = ClientStatusMsg.decode(message.put.body.bin);

			rssiData[clientStatusMsg.clientUuid] = clientStatusMsg.rxSignalStrDb;

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "client-status-update", "message": clientStatusMsg }));
			})
		} else if (message.put.head.path == "/event/lcdStatus") {
			const lcdStatusMessage = Err.decode(message.put.body.bin);
			// For real-time updates on the prices page
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "lcd-status-update", "message": lcdStatusMessage }));
			})
		} else if (message.put.head.path == "/event/getRadioConfigStatus") {
			let clientRadioConfig = ClientRadioConfigMsg.decode(message.put.body.bin);
			// Fill in defaults, since protobuf doesn't transmit those
			if (!clientRadioConfig.hasOwnProperty("networkId")) {
				clientRadioConfig.networkId = 0;
			}
			if (!clientRadioConfig.hasOwnProperty("channel")) {
				clientRadioConfig.channel = 0;
			}
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "client-radio-config", "message": { type: 'get', data: clientRadioConfig} }));
			})
		} else if (message.put.head.path == "/event/setRadioConfigStatus") {
			const msg = ClientRadioConfigMsg.decode(message.put.body.bin);

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "client-radio-config", "message": { type: 'set', data: { success: msg.success }} }));
			})
		} else if (message.put.head.path == "/event/radioDetectionStatus") {
			const status = Err.decode(message.put.body.bin);
			//console.log(status);
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "detect-displays", "message": status }));
			})
		} else if (message.put.head.path == "/event/logUpdate") {
			const baseMsg = BaseMsg.decode(message.put.body.bin);

			wss.clients.forEach(function (client) {
                client.send(JSON.stringify({ "topic": "log-monitor", "message": {"logMsg": baseMsg.description, "level": baseMsg.value, "devLog": baseMsg.status }}));
			})
		} else if (message.put.head.path == "/event/modStatusDisplay") {
			const display = Display.decode(message.put.body.bin);

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "mod-status-display", "message": JSON.stringify(display) }));
			})
		} else if (message.put.head.path == "/event/networkStatus") {
			const network_status = NetworkStatusReport.decode(message.put.body.bin);

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "network-status", "message": JSON.stringify(network_status.statusPacket) }));
			})
		} else if (message.put.head.path == "/event/networkPriceUpdateStatus") {
			const network_price_status = BaseMsg.decode(message.put.body.bin);

            let network_price_update_success = false;
            if (network_price_status.hasOwnProperty("status")) {
                network_price_update_success = true;
            }
        
			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "network-price-update-status", "message": { "success": network_price_update_success } }));
			})
		} else if (message.put.head.path == "/event/appStatusUpdate") {
			const appStateMsg = AppStates.decode(message.put.body.bin);
            console_log_warn("appStatusUpdate" + JSON.stringify(appStateMsg));

			wss.clients.forEach(function (client) {
				client.send(JSON.stringify({ "topic": "app-status-update", "message": appStateMsg }));
			})
		}
    }

	wss.on('connection', (socket) => {
        logToUI('Websocket client connected', true, logLevels.Extreme, false);

		socket.on('message', (message) => {
			console.log(`Received message: ${message}`);
		});

		socket.on('close', () => {
			console.log('Client disconnected');
		});

	});

	await new Promise(resolve => setTimeout(resolve, 1000)); // Need to give time for the subscription to get sent to the publisher

    await getDataZmq('/pos').then( (fl3_prod_data) => {  products = fl3_prod_data; });
}

function logToUI(logMsg, console_log = true, level = logLevels.Info, log_to_ui = true) {
	const date = new Date();
	
	const dateStr = `${date.toLocaleDateString("en-US", {year: "numeric", month: "2-digit", day: "2-digit"})} ${String(date.getHours()).padStart(2, '0')}:${String(date.getMinutes()).padStart(2, '0')}:${String(date.getSeconds()).padStart(2, '0')}`;
	logMsg = dateStr + ": " + logMsg;

	if (console_log) {
		if (level == logLevels.Extreme) {
			console_log_magenta(logMsg);
		} else if (level == logLevels.Debug) {
			console_log_blue(logMsg);
		} else if (level == logLevels.Verbose) {
			console_log_cyan(logMsg);
		} else if (level == logLevels.Info) {
			console_log_ok(logMsg);
		} else if (level == logLevels.Warning) {
			console_log_warn(logMsg);
		} else if (level == logLevels.Error) {
			console_log_error(logMsg);
		}
	}

	wss.clients.forEach(function (client) {
		client.send(JSON.stringify({ "topic": "log-monitor", "message": {"logMsg": logMsg, "level": level, "devLog": !log_to_ui }}));
	})
}

const endpoint_whitelist = [ "/network_status" ];
// ------ Public Functions ------
async function getDataZmq(endpoint) {
    const reply = await getFromBackEnd(endpoint);
    if (!reply || !reply.response || !reply.response.body || !reply.response.body.txt)
    {
        if (!endpoint_whitelist.includes(endpoint)) {
            logToUI(`endpoint ${endpoint}: No data in the response`, true, logLevels.Warning, false);
        }
        return {}
    }
    let data = JSON.parse(reply.response.body.txt);
    return data;
}

async function putDataZmq(endpoint, data) {
    const reply = await sendToBackEnd(endpoint, data);
    if (!reply || !reply.response || !reply.response.body || !reply.response.body.txt)
    {
        return {}
    }
    let rep_data = JSON.parse(reply.response.body.txt);
    return rep_data;
}

function getMetaData() {
	return metaData;
}

function getRSSIData() {
	return rssiData;
}

function getPingData() {
	return pingData;
}

function setPingData(jsonData) {
	pingData = jsonData;
}

function getWifiNetworks() {
	return wifiNetworks;
}

function setWifiNetworks(networks) {
	wifiNetworks = networks;
}

let logBlacklist = ["/log_to_file"]
// Write the price data to the backend socket (J. Burke)
async function sendToBackEnd(path, data) {
	const buffer = Message.encode(
		{
			"put": {
				"head": {
					"to": [],
					"from": {
						"pid": process.pid,
						"tag": ""
					},
					"path": path,
					"format": "Json",
					"params": [
						{
							"key": "format",
							"value": "json"
						}
					],
					"headers": [],
					"meta": [
						{
							"key": "ttl",
							"value": "PT3S"
						}
					]
				},
				"body": {
					"txt": JSON.stringify(data)
				}
			}
		}
	).finish();
    if (!logBlacklist.includes(path)) {
        logToUI(`ZeroMQ PUT at ${path}`, true, logLevels.Verbose, false);
    }

    const release = await mutex.acquire();

    let result;
    try {
        await reqSocket.send(buffer);
        [result] = await reqSocket.receive();
    } catch (error) {
        console.error(`error sending request for ${path}`, error);
        await reqSocket.close();
        reqSocket = new zmq.Request({receiveTimeout: 10000});
        await reqSocket.connect("ipc:///run/vega/fl5/zmq.sock");
    } finally {
        release();
    }

    if (result) {
        return Message.decode(result);
    } else {
        return [];
    }
}

async function getFromBackEnd(path) {
	const buffer = Message.encode(
		{
			"get": {
				"head": {
					"to": [],
					"from": {
						"pid": process.pid,
						"tag": ""
					},
					"path": path,
					"format": "Json",
					"params": [
						{
							"key": "format",
							"value": "json"
						}
					],
					"headers": [],
					"meta": [
						{
							"key": "ttl",
							"value": "PT3S"
						}
					]
				}
			}
		}
	).finish();
    logToUI(`ZeroMQ GET at ${path}`, true, logLevels.Extreme, false);

    const release = await mutex.acquire();

    let result;
    try {
        await reqSocket.send(buffer);
        [result] = await reqSocket.receive();
    } catch (error) {
        console.error(`error sending request for ${path}`, error);
        await reqSocket.close();
        reqSocket = new zmq.Request({receiveTimeout: 10000});
        await reqSocket.connect("ipc:///run/vega/fl5/zmq.sock");
    } finally {
        release();
    }

    if (result) {
        return Message.decode(result);
    } else {
        return [];
    }
}

function getSWVersion() {
	if (fs.existsSync(versionFile)) {
        const versFileLines = fs.readFileSync(versionFile, { encoding: 'utf8', flag: 'r' });
		metaData.vega_sw_version = versFileLines.split("\n")[0];
	} else {
        logToUI(`Version file at ${versionFile} does not exist`, true, logLevels.Error, false);
	}
}

function getModFWVersion() {
	if (fs.existsSync(fudFile)) {
		let fudText = fs.readFileSync(fudFile, 'utf8').toString();
		let correctLine = fudText.substring(fudText.indexOf(":4029C0"), fudText.indexOf(":402A00")); // Should always get the line that the version is on
		let ddchecksumIndex = correctLine.indexOf("36C7");
		if (ddchecksumIndex > 0) { // if the dd checksum isn't in this block, then don't go looking for the fw version
			let versionStartIndex = ddchecksumIndex + 4; // dd checksum is 4 characters, the version number should always be here
			let versionString = correctLine[versionStartIndex + 3] + correctLine[versionStartIndex] + correctLine[versionStartIndex + 1]; // Get the last 3 characters (cause that's what the mods display) stored as little-endian
			metaData.mod_fw_version = versionString;
		}
	} else {
        logToUI(`Version file at ${versionFile} does not exist`, true, logLevels.Error, false);
	}
}

function getFontFileVersion() {
	if (fs.existsSync(fontFile)) {
		let fontText = fs.readFileSync(fontFile, 'utf8').toString();
		// TODO: Find out where in the file the font version is kept
	} else {
        logToUI(`Version file at ${versionFile} does not exist`, true, logLevels.Error, false);
	}
}

function getVersionInfo() {
	getSWVersion();
	getModFWVersion();
	getFontFileVersion();
}
